/**************************************************************************
           fmifstream.cpp  -  providing fms-file reading utility
                             -------------------
    begin                : April 10th 2002
    copyright            : (C) 2002-2003 by Daniel Gruen
    email                : daniel_gruen@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 
#include <cmath>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iomanip>

#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "include/fmifstream.h"
#include "include/version.h"

// #define fileformat_debug

using namespace std;

void FMData::init(char *filename) {

	int fd;

	if(debug_level)
		cout << filename << " is going to be opened, name length: " << strlen(filename) << endl;

    try {
	fd = open(filename, O_RDONLY);

	if(debug_level)
		cout << "opened " << filename << ", fd=" << fd << endl;

	if(fd < 1) {
		if(debug_level)
			cout << "fmifstream: couldn't open " << filename << endl;
		throw 1;
	}

	if(read(fd, buf, 1) != 1) {
		if(debug_level)
			cout << "fmifstream: couldn't read 1" << endl;
		throw 2;
	}
#ifdef fileformat_debug
	cout << "ffd: read mode " << int(buf[0]) << endl;
#endif
	mode = buf[0];
	
	if(debug_level)
		cout << "mode: " << int(mode) << endl;

	if(mode > 3){
		if(debug_level)
			cout << "fmifstream: wrong mode!" << endl;
		throw 2;
	}

	if(read(fd, buf, 1) != 1) {
		if(debug_level)
			cout << "fmifstream: couldn't read 2" << endl;
		throw 2;
	}
	nlen = buf[0];
#ifdef fileformat_debug
	cout << "ffd: read nlen " << int(buf[0]) << endl;
#endif

	name = new char[nlen+1];
	if(read(fd, name, nlen) != nlen) {
		if(debug_level)
			cout << "fmifstream: couldn't read 3" << endl;
		throw 2;
	}
	name[nlen] = '\0';
#ifdef fileformat_debug
	cout << "ffd: read name " << name << endl;
#endif

	if(mode == 0) {   // bare values

		if(read(fd, buf, 3) != 3) {
			if(debug_level)
				cout << "fmifstream: couldn't read 4a" << endl;
			throw 2;
		}
		vlen = buf[0] * 65536 + buf[1] * 256 + buf[2];

		if(vlen<2) {
			if(debug_level)
				cout << "fmifstream: vlen too small 1" << endl;
			throw 2;
			vlen = 2;
		}

		if(read(fd, buf, 1) != 1) {
			if(debug_level)
				cout << "fmifstream: couldn't read 5a" << endl;
			throw 2;
		}
		bytes = buf[0];
		if(bytes != 1){
			if(debug_level)
				cout << "fmifstream: bytes wrong 1" << endl;
			throw 2;
		}

		val = new unsigned char[vlen];
		if(read(fd, val, vlen) != vlen) {
			if(debug_level)
				cout << "fmifstream: couldn't read 6a" << endl;
			throw 2;
		}
	}

	if(mode == 1) {  // Functions

		if(read(fd, buf, 3) != 3) {
			if(debug_level)
				cout << "fmifstream: couldn't read 4b" << endl;
			throw 2;
		}
		vlen = buf[0] * 65536 + buf[1] * 256 + buf[2];
		if(debug_level>1)
			cout << "fmifstream: vlen=" << vlen << endl;

		if(vlen < 2) {
			if(debug_level)
				cout << "fmifstream: vlen too small 2" << endl;
			throw 2;
			vlen = 2;
		}

		if(read(fd, buf, 1) != 1) {
			if(debug_level)
				cout << "fmifstream: couldn't read 5b" << endl;
			throw 2;
		}
		bytes = buf[0];
		if(bytes != 1){
			if(debug_level)
				cout << "fmifstream: bytes wrong 2" << endl;
			throw 2;
		}
		if(debug_level>1)
			cout << "fmifstream: bytes=" << bytes << endl;

		vtl = new VTL[vlen];
		for(i = 0; i < vlen; i++) {
			if(read(fd, buf, 4) != 4){
				if(debug_level)
					cout << "fmifstream: couldn't read 6b" << endl;
				throw 2;
			}
			vtl[i].setTime(buf[0] * 256 + buf[1]);
			if(i)
				vtl[i].setTime(vtl[i].getTime() + vtl[i-1].getTime() + 1);
			if(debug_level>1)
				cout << "fmifstream: vtl[" << i << "].time=" << vtl[i].getTime() << endl;
			vtl[i].setValue(buf[2]);
			if(debug_level>1)
				cout << "fmifstream: vtl[" << i << "].value=" << int(vtl[i].getValue()) << endl;
			vtl[i].setLinetype(buf[3]);
			if(debug_level>1)
				cout << "fmifstream: vtl[" << i << "].linetype=" << int(vtl[i].getLinetype()) << endl;
			if(vtl[i].getLinetype() == 4){
				if(read(fd, buf, 1) != 1){
					if(debug_level)
						cout << "fmifstream: couldn't read 7b" << endl;
					throw 2;
				}
				vtl[i].setAttr1(-buf[0]/64.0);
				if(debug_level>1)
					cout << "fmifstream: vtl[" << i << "].attr1=" << vtl[i].getAttr1() << endl;
			}
			if(vtl[i].getLinetype() == 5){
				if(read(fd, buf, 1) != 1){
					if(debug_level)
						cout << "fmifstream: couldn't read 8b" << endl;
					throw 2;
				}
				vtl[i].setAttr1(buf[0]/64.0);
				if(debug_level>1)
					cout << "fmifstream: vtl[" << i << "].attr1=" << vtl[i].getAttr1() << bytes << endl;
			}
			if(vtl[i].getLinetype() == 6 || vtl[i].getLinetype() == 7){
				if(read(fd, buf, 1) != 1){
					if(debug_level)
						cout << "fmifstream: couldn't read 9b" << endl;
					throw 2;
				}
				vtl[i].setAttr1(buf[0] % 16 - 8);
				if(debug_level>1)
					cout << "fmifstream: vtl[" << i << "].attr1=" << vtl[i].getAttr1() << bytes << endl;
				vtl[i].setAttr2((buf[0] - buf[0] % 16) / 16 - 7);
				if(debug_level>1)
					cout << "fmifstream: vtl[" << i << "].attr2=" << vtl[i].getAttr2() << bytes << endl;
				if(abs(vtl[i].getAttr1()) == abs(vtl[i].getAttr2())){
					if(read(fd, buf, 1) != 1){
						if(debug_level)
							cout << "fmifstream: couldn't read 10b" << endl;
						throw 2;
					}
					vtl[i].setAttr3(buf[0]);
					if(debug_level>1)
						cout << "fmifstream: vtl[" << i << "].attr3=" << vtl[i].getAttr3() << bytes << endl;
				}
			}

			if(vtl[i].getLinetype() == 0 && i != vlen - 1){
				if(debug_level)
					cout << "fmifstream: unexpected but correct end of values" << endl;
				throw 2;
			}
			if(vtl[i].getLinetype() > 7){
				if(debug_level)
					cout << "fmifstream: unknown linetype" << endl;
				throw 2;
			}
			}
	}

	if(mode == 2) { // Midi-like

		if(read(fd, buf, 2) != 2) {
			cout << "can't read 2.1" << endl;
			throw 2;
		}
		vlen = buf[0] * 256 + buf[1];
#ifdef debug_mode
		cout << "VLEN: " << vlen << endl;
#endif // debug_mode

		if(vlen<2) {
			if(debug_level)
				cout << "fmifstream: vlen too small 3" << endl;
			throw 2;
			vlen = 2;
		}

		pbs = new PBS[vlen];

		//double q = pow(10, log10(2.0) / 12.0); //somebody else now does this for us

		bytes = 0;

		for(i = 0; i < vlen; i++) {

			if(read(fd, buf, 2) != 2){
				cout << "can't read 2.2" << endl;
				// I guess the file is over
				// maybe we can still use it
				vlen = i;
				goto skip_read;
			}

			pbs[i].pitch = pitch(buf[0]);
#ifdef debug_mode
			cout << "PITCH: " << int(buf[0]) % 128 << endl;
#endif // debug_mode
			//pbs[i].pitch = buf[0]%128;
			//pbs[i].pitch = pow(q, int(pbs[i].pitch + 3)) / 64.0; // * pow(2.0, (pbs[i].pitch - int(pbs[i].pitch) % 12) / 12);

			//if(buf[0]%128 == 120) // if it's a rest
			//	pbs[i].pitch = 0;

			if(buf[0] < 128)
				pbs[i].beats=beats(buf[1],0);
			else
				pbs[i].beats=beats(buf[1],1);
#ifdef debug_mode
			cout << "BEATS: " << buf[1] % 8 << endl;
#endif // debug_mode
			//pbs[i].beats=beats(buf[1],buf[0]>127);

			pbs[i].style = (buf[1] - buf[1] % 8) / 8;
#ifdef debug_mode
			cout << "STYLE: " << (buf[1] - buf[1] % 8) / 8 << endl;
#endif // debug_mode
			pbs[i].style=0; // es bricht mir den Hals, wenn ich obige Zeile verwende
			if(pbs[i].style == 31){ // it's a chord
#ifdef debug_mode
				cout << "CHORD FOUND" << endl;
#endif // debug_mode
				if(read(fd, buf, 1) != 1){
					throw 2;
				}
#ifdef debug_mode
				cout << "READ: " << int(buf[0]) << endl;
#endif // debug_mode
				pbs[i].chord = buf[0];
			}
			if(debug_level)
				cout << "read = pitch: " << setw(9) << pbs[i].pitch << ", style: " << setw(3) << pbs[i].style << ", beats: " << setw(5) << pbs[i].beats << endl;
		}

		skip_read:

#ifdef debug_mode
		cout << "summary:" << endl;
#endif // debug_mode
		for(i=0; i<vlen; i++){
#ifdef debug_mode
		cout << pbs[i].pitch << " " << pbs[i].style << " " << pbs[i].beats << endl;
#endif // debug_mode
		}

	}
	if(mode == 3) { // Fourier

		bool usephase = 0;

		if(read(fd, buf, 2) != 2) {
			if(debug_level)
				cout << "fmifstream: couldn't read 4a" << endl;
			throw 2;
		}
		vlen = buf[0] * 256 + buf[1];

		if(vlen<1) {
			if(debug_level)
				cout << "fmifstream: vlen=" << vlen << " too small 4" << endl;
			throw 2;
			vlen = 1;
		}

		if(read(fd, buf, 1) != 1) {
			if(debug_level)
				cout << "fmifstream: couldn't read 5a" << endl;
			throw 2;
		}
		bytes = buf[0];

		if(bytes>128) {
			bytes-=128;
			usephase=1;
			if(debug_level>1)
				cout << "fmifstream: using phase" << endl;
		}

		fa = new FAP[vlen];
		float gamp = 0.0;
		for(int i=0; i<vlen; i++){
			fa[i].amp = 0.0;
			for(int j=0; j<bytes; j++){
				if(read(fd, buf, 1) != 1) {
					if(debug_level)
						cout << "fmifstream: couldn't read 6a" << endl;
					throw 2;
				}
				fa[i].amp+=float(buf[0])*pow(256.0,bytes-j-1);
			}
			if(usephase){
				if(read(fd, buf, 1) != 1) {
					if(debug_level)
						cout << "fmifstream: couldn't read 7a" << endl;
					throw 2;
				}
				fa[i].phase=float(buf[0]/256.0*180.0);
			}
			else
				fa[i].phase=0.0;
			fa[i].amp/=pow(256.0,bytes);
			if(debug_level>1)
					cout << "read amplitude " << i << ": " << fa[i].amp << endl;
			gamp += fa[i].amp;
		}
		for(int i=0; i<vlen; i++)
			fa[i].amp = fa[i].amp / gamp;
	}
    }
    catch(...) {
    	throw;
    }
}


void FMData::cleanup() {

	if(debug_level >= 1)
		cout << "cleaning up for " << name << endl;

	if(strcmp(name,"---")){
		delete []name;
	}


	switch(mode) {
	case 0:
		delete []val;
		break;
	case 1:
		delete []vtl;
		break;
	case 2:
		delete []pbs;
		break;
	}

	if(debug_level >= 1)
		cout << "cleanup is clean" << endl;

}

float pitch(char c) {
	if(c%128==120)
		return 0.0;
	return pow(pow(10, log10(2.0) / 12.0), int((c%128) + 3)) / 64.0;
}

float beats(char c, bool third) {
int n = c%8; if(n<0)n+=8;
if(!third)
{
	switch(n){
	    case 0:
		return 1.0 / 16.0;
		break;
	    case 1:
		return 1.0 / 8.0;
		break;
	    case 2:
		return 3.0 / 16.0;
		break;
	    case 3:
		return 1.0 / 4.0;
		break;
	    case 4:
		return 3.0 / 8.0;
		break;
	    case 5:
		return 1.0 / 2.0;
		break;
	    case 6:
		return 3.0 / 4.0;
		break;
	    case 7:
		return 1.0;
		break;
	}
}
switch(n){
	case 0:
	return 1.0 / 32.0;
	break;
    case 1:
	return 1.0 / 12.0;
	break;
    case 2:
	return 1.0 / 10.0;
	break;
    case 3:
	return 1.0 / 9.0;
	break;
    case 4:
	return 3.0 / 2.0;
	break;
    case 5:
	return 1.0 / 6.0;
	break;
    case 6:
	return 1.0 / 5.0;
	break;
    case 7:
	return 1.0 / 3.0;
	break;
}
return 1;
}
