// This is Free Software
// See the GNU General Public License @ http://www.gnu.org/copyleft/gpl.html
// for details about the terms and conditions for using and distributing
// this piece of software.
// This software comes with absolutely NO WARRANTY!
// (c) 2005 by Daniel Gruen <daniel_gruen@web.de>

#include <qpainter.h>
#include <qpixmap.h>
#include <qhbox.h>
#include <qslider.h>
#include <qlabel.h>
#include <qlcdnumber.h>
#include <qspinbox.h>
#include <qpushbutton.h>
#include <qpen.h>
#include <qcolor.h>
#include <qpalette.h>
#include <qcheckbox.h>
#include <qwidget.h>
#include <qtimer.h>

#include <cmath>
#include <ctime>
#include <cstdlib>

#include <unistd.h>

#include "qgoodvibes.h"


QCanvas *GVCanvas;
int   xang,yang;
const dddcoord null={0,0,0}; // Nullvektor

const int dist = 100;
const int coupling_dist = 90;
const int xos = 200;
const int yos = 75;
const int zos = 500;
const int frame_t = 10; // time per frame (ms)
const int k = 400;
const int dim = 5; // number of nodes in each direction
const bool friction = FALSE;

const double velocity_factor = 0.05;
const double acceleration_factor = 0.1;

// projects xyz-coords on xy
ddcoord xycoord(dddcoord a, float k) {
   ddcoord tmp;
   tmp.x = int(a.x) + (400 - int(a.x)) * (0.5-(k/(a.z+k)));
   tmp.y = int(a.y) + (300 - int(a.y)) * (0.5-(k/(a.z+k)));
   return tmp;
}

// turns xyz-coords around an axis
// special thanks to Mr. M. Deffner
dddcoord xzturn(dddcoord a , int W)
{
   float D=atan(a.x/a.z);             //                  { Ausgangswinkel }
   float G=D+(pi/180)*W;             //                    { Drehwinkel }
   if (a.z<0) G=G+(pi);
    float R=sqrt(a.z*a.z+a.x*a.x); //                          { Radius }
   a.z=int(R*cos(G));
   a.x=int(R*sin(G));
   return a;
}

dddcoord yzturn(dddcoord a , int W)
{
   float D=atan(a.y/a.z);             //                  { Ausgangswinkel }
   float G=D+(pi/180)*W;             //                    { Drehwinkel }
   if (a.z<0) G=G+(pi);
    float R=sqrt(a.z*a.z+a.y*a.y); //                          { Radius }
   a.z=int(R*cos(G));
   a.y=int(R*sin(G));
   return a;
}

dddcoord vadd(dddcoord a, dddcoord b)
{
   dddcoord x = {a.x+b.x, a.y+b.y, a.z+b.z};
   return x;
}

dddcoord vsub(dddcoord a, dddcoord b)
{
   dddcoord x = {a.x-b.x, a.y-b.y, a.z-b.z};
   return x;
}

dddcoord vmult(dddcoord a, double f)
{
   dddcoord x = {a.x*f, a.y*f, a.z*f};
   return x;
}

double vabs(dddcoord a)
{
   return sqrt(a.x*a.x + a.y*a.y + a.z*a.z);
}

dddcoord normalize(dddcoord a)
{
   return vmult(a,1.0/vabs(a));
}

// constructor of QLiss3D-Widget
QGoodVibes::QGoodVibes(QWidget *parent, const char *name) : QCanvasView(parent, name) {

	GVCanvas = new QCanvas;
	
	setCanvas(GVCanvas);
	xang = 0;  // viewing angle = 0
	yang = 0;
	msx = msy = 0; // mouse coordinates when dragging
	
	setFixedSize(800,600);
	GVCanvas->resize(780,580);
	srand(time((time_t *)NULL));
	setFocusPolicy(QWidget::StrongFocus);
	setFocus();
	
	QTimer *timer = new QTimer(this);
        timer->start( frame_t, FALSE ); // 0.5 s
	
	dddcoord dp;
	for(int i=0; i<dim; i++) // initialize Nodes
	{
	  dp.x=i*dist+xos;
	  for(int j=0; j<dim; j++)
	  {
	    dp.y=j*dist+yos;  
		for(int k=0; k<dim; k++)
		{
		  block[i][j][k] = new GVNode;
		  dp.z=k*dist+zos;
		  block[i][j][k]->position=dp;
		  block[i][j][k]->velocity=null; // no velocity
		  block[i][j][k]->mass=1;
		  block[i][j][k]->firstC=0;
		  
		  if(i)
		    block[i][j][k]->setCoupling(block[i-1][j][k],1);
		  if(j)
		    block[i][j][k]->setCoupling(block[i][j-1][k],1);
		  if(k)
		    block[i][j][k]->setCoupling(block[i][j][k-1],1);
		    
		    connect( timer, SIGNAL(timeout()), block[i][j][k], SLOT(animate()) );
		}
	  }
	}
	connect( timer, SIGNAL(timeout()), GVCanvas, SLOT(update()) );
	connect( timer, SIGNAL(timeout()), this, SLOT(debug_update()));
	show();
}

// set turning angles
void QGoodVibes::setXAngle(int degrees){
	if(degrees>180)
		degrees=-180;
	if(degrees<-180)
		degrees=180;
	if(xang==degrees)
		return;
	xang = degrees;
	emit xAngleChanged(xang);
}

void QGoodVibes::setYAngle(int degrees){
	if(degrees>180)
		degrees=-180;
	if(degrees<-180)
		degrees=180;
	if(yang==degrees)
		return;
	yang = degrees;
	emit yAngleChanged(yang);
}

void QGoodVibes::keyPressEvent(QKeyEvent * e){
	switch(e->key()){
		case Qt::Key_Left:
			setXAngle(xang+1);
			break;
		case Qt::Key_Right:
			setXAngle(xang-1);
			break;
		default:
			e->ignore();
	}
}

void QGoodVibes::mousePressEvent(QMouseEvent * e){
	msx = e->x();
	msy = e->y();
}

// turn it following the mouse
void QGoodVibes::mouseMoveEvent(QMouseEvent * e){
	setXAngle(xang-msx+e->x());
	setYAngle(yang+msy-e->y());
	msx = e->x();
	msy = e->y();
}

/**************************************************************/

GVNode::GVNode()
  : QObject(), QCanvasEllipse(3,3,GVCanvas)
{
	position = velocity = null;
	mass = 1;
	firstC=0;
	setX(100);
	setY(100);
	setBrush(QColor(255,50,100));
	show();
}

void GVNode::setCoupling(GVNode *c, double k)
{
	//cout << "setting coupling... " << flush;
	GVCoupling *tmpc = new GVCoupling;

	tmpc->n=c;
	tmpc->k=k;
	tmpc->d=coupling_dist;
	//tmpc->next=0;
	
	addCoupling(tmpc);
	
	GVCoupling *tmpc2 = new GVCoupling;
	tmpc2->n=this;
	tmpc2->k=k;
	tmpc2->d=coupling_dist;

	c->addCoupling(tmpc2); // couple partner as well
	//cout << " finished!" << endl;
}

void GVNode::addCoupling(GVCoupling *c)
{

	if(!firstC)
	{
		//cout << " have to set firstC..." << flush;
		firstC = c;
		//firstC->next=0;
	}
	else
	{  
	  GVCoupling *lastC = firstC;
	  //cout << " have to find lastC..." << flush;
	  while(lastC->nextC())
	  {
	  	//cout << int(lastC) << endl;
		lastC = lastC->nextC();
		//sleep(1);
	  }
	  //cout << " found it! " << flush;
	  lastC->setNextC(c);
	}
	//cout << "successfull..." << flush;
}

void GVNode::animate()
{
	// draw positions
	dddcoord tmpp = position;
	tmpp.x -= 400;
	tmpp.z -= 50;
	tmpp = xzturn(tmpp, xang);
	tmpp.x += 400;
	tmpp.z += 240;
	
        tmpp.y -= 300;
	tmpp.z -= 250;
	tmpp = yzturn(tmpp, yang);
	tmpp.y += 300;
	tmpp.z += 440;
	
	ddcoord p = xycoord(tmpp,k); // projection
	setX(p.x);
	setY(p.y);
	//cout << "p.x=" << p.x << ",p.y=" << p.y << endl;
	
	// change positions
	position.x += velocity.x * velocity_factor;
	position.y += velocity.y * velocity_factor;
	position.z += velocity.z * velocity_factor;
	
	// change velocities
	dddcoord f={0,0,0}; // resulting force
	GVCoupling * tmpc = firstC;
	while(tmpc)
	{
		// direction of force
		dddcoord df = vsub(tmpc->n->position, position);
		// strength (absolute force)
		double s = tmpc->k*(vabs(df)-tmpc->d);
		df = normalize(df);
		f = vadd(f, vmult(df,s)); 
		tmpc = tmpc->nextC();
	}
	velocity.x += f.x * acceleration_factor / mass;
	velocity.y += f.y * acceleration_factor / mass;
	velocity.z += f.z * acceleration_factor / mass;
	
	if(friction)
	  velocity = vmult(velocity,0.995); // friction
	
	show();
}
